unit basszxtuneMain;

interface

uses
  Windows,
  AnsiStrings,
  SysUtils,
  Math,
  apiObjects,
  apiCore,
  apiFileManager,
  apiDecoders,
  apiPlugin;

{$REGION 'BASS API'}
const
  BASS_POS_BYTE = 0;
  BASS_SAMPLE_8BITS       = 1;   // 8 bit
  BASS_SAMPLE_FLOAT       = 256; // 32 bit floating-point
  BASS_STREAM_DECODE      = $200000;// don't play the stream, only decode (BASS_ChannelGetData)
  BASS_TAG_OGG            = 2; // OGG comments : series of null-terminated UTF-8 strings
  BASS_UNICODE            = $80000000;

type
  HPLUGIN = DWORD;
  HSTREAM = DWORD;      // sample stream handle
  HSAMPLE = DWORD;      // sample handle
  QWORD = Int64;

  BASS_CHANNELINFO = record
    freq: DWORD;        // default playback rate
    chans: DWORD;       // channels
    flags: DWORD;       // BASS_SAMPLE/STREAM/MUSIC/SPEAKER flags
    ctype: DWORD;       // type of channel
    origres: DWORD;     // original resolution
    plugin: HPLUGIN;    // plugin
    sample: HSAMPLE;    // sample
    {$IFDEF CPUX64}
    padding: DWORD;
    {$ENDIF}
    filename: PChar;    // filename
  end;

  TBASSChannelGetTagsFunc = function (handle: HSTREAM; tags: DWORD): Pointer; stdcall;
  TBassStreamBytesToSecondsFunc = function (handle: DWORD; pos: QWORD): Double; stdcall;
  TBassStreamCreateFileFunc = function (Mem: BOOL; f: Pointer; offset, length: QWORD; flags: DWORD): HSTREAM; stdcall;
  TBassStreamFreeFunc = function (handle: HSTREAM): BOOL; stdcall;
  TBassStreamGetDataFunc = function (handle: DWORD; buffer: Pointer; length: DWORD): DWORD; stdcall;
  TBassStreamGetInfoFunc = function (handle: DWORD; var info: BASS_CHANNELINFO): BOOL; stdcall;
  TBassStreamGetLengthFunc = function (handle, mode: DWORD): QWORD; stdcall;
  TBassStreamGetPositionFunc = function (handle, mode: DWORD): QWORD; stdcall;
  TBassStreamSetPositionFunc = function (handle: DWORD; pos: QWORD; mode: DWORD): BOOL; stdcall;

  { IBASSZXTuneAPI }

  IBASSZXTuneAPI = interface
  ['{3EAA82B8-919B-4B65-83EB-F9D81B0F714D}']
    function StreamCreate(FileName: PWideChar): HSTREAM; stdcall;
    function StreamGetData(handle: DWORD; buffer: Pointer; length: DWORD): DWORD; stdcall;
    function StreamGetDuration(handle: DWORD): Single; stdcall;
    function StreamGetInfo(handle: DWORD; var info: BASS_CHANNELINFO): BOOL; stdcall;
    function StreamGetLength(handle, mode: DWORD): QWORD; stdcall;
    function StreamGetPosition(handle, mode: DWORD): QWORD; stdcall;
    function StreamGetTags(handle: DWORD): PAnsiChar; stdcall;
    function StreamSetPosition(handle: DWORD; pos: QWORD; mode: DWORD): BOOL; stdcall;
    procedure StreamFree(AHandle: HSTREAM); stdcall;
  end;

  { TBASSZXTuneAPI }

  TBASSZXTuneAPI = class(TInterfacedObject, IBASSZXTuneAPI)
  strict private
    FLibHandle: THandle;
    FStreamBytesToSecondsFunc: TBassStreamBytesToSecondsFunc;
    FStreamCreateFileFunc: TBassStreamCreateFileFunc;
    FStreamFreeFunc: TBassStreamFreeFunc;
    FStreamGetDataFunc: TBassStreamGetDataFunc;
    FStreamGetInfoFunc: TBassStreamGetInfoFunc;
    FStreamGetLengthFunc: TBassStreamGetLengthFunc;
    FStreamGetPositionFunc: TBassStreamGetPositionFunc;
    FStreamGetTagsFunc: TBASSChannelGetTagsFunc;
    FStreamSetPositionFunc: TBassStreamSetPositionFunc;

    function SafeGetProcAddress(ALibHandle: THandle; const AProcName: PChar): Pointer;
  public
    constructor Create(ALibHandle: THandle);
    destructor Destroy; override;
    // IBASSZXTuneAPI
    function StreamCreate(FileName: PWideChar): HSTREAM; stdcall;
    function StreamGetData(handle: DWORD; buffer: Pointer; length: DWORD): DWORD; stdcall;
    function StreamGetDuration(handle: DWORD): Single; stdcall;
    function StreamGetInfo(handle: DWORD; var info: BASS_CHANNELINFO): BOOL; stdcall;
    function StreamGetLength(handle, mode: DWORD): QWORD; stdcall;
    function StreamGetPosition(handle, mode: DWORD): QWORD; stdcall;
    function StreamGetTags(handle: DWORD): PAnsiChar; stdcall;
    function StreamSetPosition(handle: DWORD; pos: QWORD; mode: DWORD): BOOL; stdcall;
    procedure StreamFree(AHandle: HSTREAM); stdcall;
  end;
{$ENDREGION}

type
  { TBASSZXTuneDecoder }

  TBASSZXTuneDecoder = class(TInterfacedObject, IAIMPAudioDecoder)
  protected
    FFinished: Boolean;
    FLibApi: IBASSZXTuneAPI;
    FHandle: THandle;
  public
    constructor Create(AHandle: THandle; ALibApi: IBASSZXTuneAPI);
    destructor Destroy; override;
    // IAIMPAudioDecoder
    function GetAvailableData: Int64; virtual; stdcall;
    function GetFileInfo(FileInfo: IAIMPFileInfo): LongBool; virtual; stdcall;
    function GetPosition: Int64; virtual; stdcall;
    function GetSize: Int64; virtual; stdcall;
    function GetStreamInfo(out SampleRate, Channels, SampleFormat: Integer): LongBool; virtual; stdcall;
    function IsRealTimeStream: LongBool; virtual; stdcall;
    function IsSeekable: LongBool; virtual; stdcall;
    function Read(Buffer: PByte; Count: Integer): Integer; virtual; stdcall;
    function SetPosition(const Value: Int64): LongBool; virtual; stdcall;
  end;

  { TBASSZXTuneFileFormats }

  TBASSZXTuneFileFormats = class(TInterfacedObject, IAIMPExtensionFileFormat)
  protected const
    SupportedFormats =
      '*.$b;*.$m;*.2sf;*.ahx;*.as0;*.asc;*.ay;*.ayc;*.bin;*.cc3;*.chi;*.cop;*.d;*.dmm;*.dsf;*.dsq;*.dst;*.esv;*.fdi;' +
      '*.ftc;*.gam;*.gamplus;*.gbs;*.gsf;*.gtr;*.gym;*.hes;*.hrm;*.hrp;*.hvl;*.kss;*.lzs;*.m;*.mod;*.msp;*.mtc;*.nsf;' +
      '*.nsfe;*.p;*.pcd;*.psc;*.psf;*.psf2;*.psg;*.psm;*.pt1;*.pt2;*.pt3;*.rmt;*.rsn;*.s;*.s98;*.sap;*.scl;*.sid;*.spc;' +
      '*.sqd;*.sqt;*.ssf;*.st1;*.st3;*.stc;*.stp;*.str;*.szx;*.td0;*.tf0;*.tfc;*.tfd;*.tfe;*.tlz;*.tlzp;*.trd;*.trs;' +
      '*.ts;*.usf;*.v2m;*.vgm;*.vgz;*.vtx;*.ym;';
  strict private
    FCore: IAIMPCore;

    function MakeString(out S: IAIMPString; const P: UnicodeString): HRESULT;
  public
    constructor Create(ACore: IAIMPCore);
    // IAIMPExtensionFileFormat
    function GetDescription(out S: IAIMPString): HRESULT; stdcall;
    function GetExtList(out S: IAIMPString): HRESULT; stdcall;
    function GetFlags(out Flags: Cardinal): HRESULT; stdcall;
  end;

  { TBASSZXTuneDecoderExtension }

  TBASSZXTuneDecoderExtension = class(TInterfacedObject, IAIMPExtensionAudioDecoderOld)
  strict private
    FLibApi: IBASSZXTuneAPI;
  public
    constructor Create(ALibApi: IBASSZXTuneAPI);
    // IAIMPExtensionAudioDecoderOld
    function CreateDecoder(FileName: IAIMPString; Flags: Cardinal;
      ErrorInfo: IAIMPErrorInfo; out Decoder: IAIMPAudioDecoder): HRESULT; stdcall;
  end;

  { TBASSZXTunePlugin }

  TBASSZXTunePlugin = class(TInterfacedObject, IAIMPPlugin)
  strict private
    FLibApi: IBASSZXTuneAPI;
  public
    // IAIMPPlugin
    function InfoGet(Index: Integer): PWideChar; stdcall;
    function InfoGetCategories: Cardinal; stdcall;
    function Initialize(Core: IAIMPCore): HRESULT; stdcall;
    procedure Finalize; stdcall;
    procedure SystemNotification(NotifyID: Integer; Data: IUnknown); stdcall;
  end;

implementation

var
  FGlobalCore: IAIMPCore = nil;

function MakeString(const S: UnicodeString): IAIMPString;
begin
  if FGlobalCore = nil then
    raise EInvalidArgument.Create('Core was not initialized');
  if Failed(FGlobalCore.CreateObject(IID_IAIMPString, Result)) then
    raise EInvalidOp.Create('Cannot create string object');
  if Failed(Result.SetData(PChar(S), Length(S))) then
    raise EInvalidOp.Create('Unable to set string');
end;

function MakeInt64(ALow, AHigh: Cardinal): Int64; inline;
begin
  Result := ALow;
  if AHigh > 0 then
    Result := (Int64(AHigh) shl 32) or Result;
end;

function FileSize(const FileName: PChar): Int64;
var
  AData: WIN32_FILE_ATTRIBUTE_DATA;
begin
  //#AI: GetFileAttributesExW works fine with locked files too
  if GetFileAttributesEx(FileName, GetFileExInfoStandard, @AData) then
    Result := MakeInt64(AData.nFileSizeLow, AData.nFileSizeHigh)
  else
    Result := 0;
end;

function WideCompareStrings(P1, P2: PWideChar; L1, L2: Integer; AIgnoreCase: Boolean = True): Integer;
const
  CaseMap: array[Boolean] of Integer = (0, NORM_IGNORECASE);
begin
  Result := CompareStringW(LOCALE_USER_DEFAULT, CaseMap[AIgnoreCase], P1, L1, P2, L2) - CSTR_EQUAL;
end;

function IsOurFileEx(const AExtsList, ATestExt: UnicodeString): Boolean;
var
  T, S: PWideChar;
  TL, SL: Integer;
begin
  Result := False;
  SL := Length(AExtsList);
  TL := Length(ATestExt);
  if (SL > 2) and (TL > 0) then
  begin
    T := @ATestExt[1];
    S := @AExtsList[1];
    while SL > TL do
    begin
      if (S^ = '*') and (PWideChar(S + TL + 1)^ = ';') then
      begin
        Result := WideCompareStrings(PWideChar(S + 1), T, TL, TL) = 0;
        if Result then Break;
      end;
      Dec(SL);
      Inc(S);
    end;
  end;
end;

function IsOurFile(const AExtsList, AFileName: UnicodeString): Boolean;
begin
  Result := IsOurFileEx(AExtsList, ExtractFileExt(AFilename));
end;


{$REGION 'BASS API'}
{ TBASSZXTuneAPI }

constructor TBASSZXTuneAPI.Create(ALibHandle: THandle);
var
  ABassHandle: THandle;
begin
  FLibHandle := ALibHandle;
  @FStreamCreateFileFunc := SafeGetProcAddress(FLibHandle, 'BASS_ZXTUNE_StreamCreateFile');

  ABassHandle := GetModuleHandle('bass.dll');
  @FStreamFreeFunc := SafeGetProcAddress(ABassHandle, 'BASS_StreamFree');
  @FStreamGetDataFunc := SafeGetProcAddress(ABassHandle, 'BASS_ChannelGetData');
  @FStreamGetInfoFunc := SafeGetProcAddress(ABassHandle, 'BASS_ChannelGetInfo');
  @FStreamGetLengthFunc := SafeGetProcAddress(ABassHandle, 'BASS_ChannelGetLength');
  @FStreamGetTagsFunc := SafeGetProcAddress(ABassHandle, 'BASS_ChannelGetTags');
  @FStreamGetPositionFunc := SafeGetProcAddress(ABassHandle, 'BASS_ChannelGetPosition');
  @FStreamSetPositionFunc := SafeGetProcAddress(ABassHandle, 'BASS_ChannelSetPosition');
  @FStreamBytesToSecondsFunc := SafeGetProcAddress(ABassHandle, 'BASS_ChannelBytes2Seconds');
end;

destructor TBASSZXTuneAPI.Destroy;
begin
  FreeLibrary(FLibHandle);
  inherited;
end;

function TBASSZXTuneAPI.SafeGetProcAddress(ALibHandle: THandle; const AProcName: PChar): Pointer;
begin
  Result := GetProcAddress(ALibHandle, AProcName);
  if Result = nil then
    raise Exception.CreateFmt('The %s was not found', [AProcName]);
end;

function TBASSZXTuneAPI.StreamCreate(FileName: PWideChar): HSTREAM;
begin
  Result := FStreamCreateFileFunc(False, FileName, 0, 0, BASS_UNICODE or BASS_STREAM_DECODE);
end;

procedure TBASSZXTuneAPI.StreamFree(AHandle: HSTREAM);
begin
  FStreamFreeFunc(AHandle);
end;

function TBASSZXTuneAPI.StreamGetData(handle: DWORD; buffer: Pointer; length: DWORD): DWORD;
begin
  Result := FStreamGetDataFunc(handle, buffer, length);
end;

function TBASSZXTuneAPI.StreamGetDuration(handle: DWORD): Single;
begin
  Result := FStreamBytesToSecondsFunc(handle, FStreamGetLengthFunc(handle, BASS_POS_BYTE));
end;

function TBASSZXTuneAPI.StreamGetInfo(handle: DWORD; var info: BASS_CHANNELINFO): BOOL;
begin
  Result := FStreamGetInfoFunc(handle, info);
end;

function TBASSZXTuneAPI.StreamGetLength(handle, mode: DWORD): QWORD;
begin
  Result := FStreamGetLengthFunc(handle, mode);
end;

function TBASSZXTuneAPI.StreamGetPosition(handle, mode: DWORD): QWORD;
begin
  Result := FStreamGetPositionFunc(handle, mode);
end;

function TBASSZXTuneAPI.StreamGetTags(handle: DWORD): PAnsiChar;
begin
  Result := FStreamGetTagsFunc(Handle, BASS_TAG_OGG);
end;

function TBASSZXTuneAPI.StreamSetPosition(handle: DWORD; pos: QWORD; mode: DWORD): BOOL;
begin
  Result := FStreamSetPositionFunc(handle, pos, mode);
end;

{$ENDREGION}

{ TBASSZXTuneDecoder }

constructor TBASSZXTuneDecoder.Create(AHandle: THandle; ALibApi: IBASSZXTuneAPI);
begin
  inherited Create;
  FLibApi := ALibApi;
  FHandle := AHandle;
end;

destructor TBASSZXTuneDecoder.Destroy;
begin
  FLibApi.StreamFree(FHandle);
  inherited Destroy;
end;

function TBASSZXTuneDecoder.GetAvailableData: Int64;
begin
  if FFinished then
    Result := 0
  else
    Result := GetSize - GetPosition;
end;

function TBASSZXTuneDecoder.GetFileInfo(FileInfo: IAIMPFileInfo): LongBool;
var
  AData: PAnsiChar;
  ADelimiter: Integer;
  AInfo: BASS_CHANNELINFO;
  AKey: AnsiString;
  APair: AnsiString;
  AValue: string;
begin
  AData := FLibApi.StreamGetTags(FHandle);
  if AData <> nil then
  begin
    while AData^ <> #0 do
    begin
      APair := AData;
      ADelimiter := AnsiStrings.PosEx('=', APair);
      AKey := Copy(APair, 1, ADelimiter - 1);
      AValue := UTF8ToString(Copy(APair, ADelimiter + 1, MaxInt));

      if AnsiSameText(AKey, 'TITLE') then
        FileInfo.SetValueAsObject(AIMP_FILEINFO_PROPID_TITLE, MakeString(AValue))
      else if AnsiSameText(AKey, 'ARTIST') then
        FileInfo.SetValueAsObject(AIMP_FILEINFO_PROPID_ARTIST, MakeString(AValue))
      else if AnsiSameText(AKey, 'DATE') then
        FileInfo.SetValueAsObject(AIMP_FILEINFO_PROPID_DATE, MakeString(AValue))
      else if AnsiSameText(AKey, 'DESCRIPTION') then
        FileInfo.SetValueAsObject(AIMP_FILEINFO_PROPID_COMMENT, MakeString(AValue))
      else if AnsiSameText(AKey, 'TYPE') then
        FileInfo.SetValueAsObject(AIMP_FILEINFO_PROPID_CODEC, MakeString(AValue))
      else if AnsiSameText(AKey, 'PROGRAM') then
        FileInfo.SetValueAsObject(AIMP_FILEINFO_PROPID_ENCODEDBY, MakeString(AValue))
      else if AnsiSameText(AKey, 'COMPUTER') then
        FileInfo.SetValueAsObject(AIMP_FILEINFO_PROPID_COPYRIGHT, MakeString(AValue));

      Inc(AData, Length(APair) + 1);
    end;
  end;

  if FLibApi.StreamGetInfo(FHandle, AInfo) then
  begin
    FileInfo.SetValueAsInt64(AIMP_FILEINFO_PROPID_FILESIZE, FileSize(AInfo.filename));
    FileInfo.SetValueAsInt32(AIMP_FILEINFO_PROPID_BITDEPTH, AInfo.origres);
    FileInfo.SetValueAsInt32(AIMP_FILEINFO_PROPID_CHANNELS, AInfo.chans);
    FileInfo.SetValueAsInt32(AIMP_FILEINFO_PROPID_SAMPLERATE, AInfo.freq);
  end;

  FileInfo.SetValueAsFloat(AIMP_FILEINFO_PROPID_DURATION, FLibApi.StreamGetDuration(fHandle));

  Result := True;
end;

function TBASSZXTuneDecoder.GetPosition: Int64;
begin
  Result := FLibApi.StreamGetPosition(FHandle, BASS_POS_BYTE);
end;

function TBASSZXTuneDecoder.GetSize: Int64;
begin
  Result := Max(0, FLibApi.StreamGetLength(FHandle, BASS_POS_BYTE));
end;

function TBASSZXTuneDecoder.GetStreamInfo(out SampleRate, Channels, SampleFormat: Integer): LongBool;
var
  AInfo: BASS_CHANNELINFO;
begin
  Result := FLibApi.StreamGetInfo(FHandle, AInfo);
  if Result then
  begin
    SampleRate := AInfo.freq;
    Channels := AInfo.chans;

    if AInfo.flags and BASS_SAMPLE_8BITS <> 0 then
      SampleFormat := AIMP_DECODER_SAMPLEFORMAT_08BIT
    else if AInfo.flags and BASS_SAMPLE_FLOAT <> 0 then
      SampleFormat := AIMP_DECODER_SAMPLEFORMAT_32BITFLOAT
    else
      SampleFormat := AIMP_DECODER_SAMPLEFORMAT_16BIT;
  end;
end;

function TBASSZXTuneDecoder.IsRealTimeStream: LongBool;
begin
  Result := False;
end;

function TBASSZXTuneDecoder.IsSeekable: LongBool;
begin
  Result := True;
end;

function TBASSZXTuneDecoder.Read(Buffer: PByte; Count: Integer): Integer;
begin
  Result := 0;
  if not FFinished and (Count > 0) then
  begin
    Result := Integer(FLibApi.StreamGetData(FHandle, Buffer, Count));
    if Result < 0 then
    begin
      FFinished := True;
      Result := 0;
    end;
  end;
end;

function TBASSZXTuneDecoder.SetPosition(const Value: Int64): LongBool;
begin
  Result := FLibApi.StreamSetPosition(FHandle, Value, BASS_POS_BYTE);
  if Result then
    FFinished := False;
end;

{ TBASSZXTuneDecoderExtension }

constructor TBASSZXTuneDecoderExtension.Create(ALibApi: IBASSZXTuneAPI);
begin
  FLibApi := ALibApi;
end;

function TBASSZXTuneDecoderExtension.CreateDecoder(FileName: IAIMPString;
  Flags: Cardinal; ErrorInfo: IAIMPErrorInfo; out Decoder: IAIMPAudioDecoder): HRESULT;
var
  AHandle: THandle;
  AFileName: UnicodeString;
begin
  try
    SetString(AFileName, FileName.GetData, FileName.GetLength);
    if IsOurFile(TBASSZXTuneFileFormats.SupportedFormats, AFileName) then
    begin
      AHandle := FLibApi.StreamCreate(PWideChar(AFileName));
      if AHandle <> 0 then
      begin
        Decoder := TBASSZXTuneDecoder.Create(AHandle, FLibAPI);
        Exit(S_OK);
      end
    end;
    Result := E_FAIL;
  except
    Result := E_UNEXPECTED;
  end;
end;

{ TBASSZXTuneFileFormats }

constructor TBASSZXTuneFileFormats.Create(ACore: IAIMPCore);
begin
  FCore := ACore;
end;

function TBASSZXTuneFileFormats.GetDescription(out S: IAIMPString): HRESULT;
begin
  Result := MakeString(S, 'ZXTune-supported module');
end;

function TBASSZXTuneFileFormats.GetExtList(out S: IAIMPString): HRESULT;
begin
  Result := MakeString(S, SupportedFormats);
end;

function TBASSZXTuneFileFormats.GetFlags(out Flags: Cardinal): HRESULT;
begin
  Flags := AIMP_SERVICE_FILEFORMATS_CATEGORY_AUDIO;
  Result := S_OK;
end;

function TBASSZXTuneFileFormats.MakeString(out S: IAIMPString; const P: UnicodeString): HRESULT;
begin
  Result := FCore.CreateObject(IID_IAIMPString, S);
  if Succeeded(Result) then
    Result := S.SetData(PChar(P), Length(P));
end;

{ TBASSZXTunePlugin }

function TBASSZXTunePlugin.InfoGet(Index: Integer): PWideChar;
begin
  case Index of
    AIMP_PLUGIN_INFO_NAME:
      Result := 'BASSZXTune-based module';
    AIMP_PLUGIN_INFO_AUTHOR:
      Result := 'Alexey Parfenov';
    AIMP_PLUGIN_INFO_SHORT_DESCRIPTION:
      Result := 'Provides support for chiptune tracker formats';
    AIMP_PLUGIN_INFO_FULL_DESCRIPTION:
      Result := TBASSZXTuneFileFormats.SupportedFormats;
  else
    Result := nil;
  end;
end;

function TBASSZXTunePlugin.InfoGetCategories: Cardinal;
begin
  Result := AIMP_PLUGIN_CATEGORY_DECODERS;
end;

function TBASSZXTunePlugin.Initialize(Core: IAIMPCore): HRESULT;
var
  ALibPath: string;
begin
  FGlobalCore := Core;
  try
    ALibPath := ExtractFilePath(GetModuleName(HInstance)) + 'basszxtune_core.dll';
    FLibApi := TBASSZXTuneAPI.Create(LoadLibrary(PChar(ALibPath)));
    Core.RegisterExtension(IID_IAIMPServiceAudioDecoders, TBASSZXTuneDecoderExtension.Create(FLibApi));
    Core.RegisterExtension(IID_IAIMPServiceFileFormats, TBASSZXTuneFileFormats.Create(Core));
    Result := S_OK;
  except
    FGlobalCore := nil;
    Result := E_FAIL;
  end;
end;

procedure TBASSZXTunePlugin.Finalize;
begin
  FGlobalCore := nil;
  FLibApi := nil;
end;

procedure TBASSZXTunePlugin.SystemNotification(NotifyID: Integer; Data: IInterface);
begin
  // do nothing
end;

end.
